
//No part of this file can be copied or released without the consent of 
//Avalanche Technology
//										
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
//										
//	Avalanche Technology Inc., Proprietary and Confidential	   *
//										
// 	Release:  1.0    Date 4/11/2022  	
//										
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
//  PART DESCRIPTION:
//
//  Technology: 22nm pMTJ STT-MRAM
//  Part:       AS302G208
//
//  Description: 2 Gigabit HIGH PERFORMANCE SERIAL PERSISTANT SRAM MEMORY
//
////////////////////////////////////////////////////////////////////////////////////
//  FILE CONTENTS : SPI/DPI/QPI RELATED TASKS 
//
////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////////
// MODULE DECLARATION                                                             //
////////////////////////////////////////////////////////////////////////////////////

`timescale 100ps / 10ps

/////////////////////////////////////////////////////////////////
//   Datasheet Timing
/////////////////////////////////////////////////////////////////
specify 
  specparam tPOR_time  = 1; 
  specparam tCSS       = 50; 
  specparam tCSH       = 40; 
`ifdef DDR_TIMING
  specparam tCH        = 125; 
  specparam tCL        = 125; 
`else
  specparam tCH        = 92.5; 
  specparam tCL        = 92.5; 
`endif
  specparam tCS        = 1300; 
  specparam tSU        = 40;
  specparam tHD        = 30;
  specparam tCO        = 90;
endspecify

/////////////////////////////////////////////////////////////////
// Select chip
/////////////////////////////////////////////////////////////////
task SelectChip;
begin
	CSn = 0;
	#(tCSS);
end
endtask 
/////////////////////////////////////////////////////////////////
// Deselect chip
/////////////////////////////////////////////////////////////////
task DeSelChip;
begin
	#(tCSH);
    	CSn = 1;
	#(tCS);
end
endtask
/////////////////////////////////////////////////////////////////
// Send byte
////////////////////////////////////////////////////////////////
task SendByte;
input [7:0] data;
integer i;
begin
  CLK = 0;

`ifdef SUHDCHK_ZHD
  for(i=0; i<8; i=i+1)
  begin
    tx[0] = 1;
    SI_IO0_reg = data[7-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    tx[0] = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end

`elsif SUHDCHK_CKFE
  for(i=0; i<8; i=i+1)
  begin
    CLK = 0;
    tx[0] = 1;
    SI_IO0_reg = data[7-i];
    #(tCL);
    CLK = 1;
    #(tCH)
    tx[0] = 0;
    CLK = 0;
  end

`else
  tx[0] = 1;
  for(i=0; i<8; i=i+1)
  begin
    SI_IO0_reg = data[7-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end
  tx[0] = 0;
`endif

end
endtask 
/////////////////////////////////////////////////////////////////
// Send Dual byte
/////////////////////////////////////////////////////////////////
task SendByteDual;
input [7:0] data;
integer i;
begin
  CLK = 0;
`ifdef SUHDCHK_ZHD
  for(i=0; i<8; i=i+2)
  begin
    tx[1:0] = 2'b11;
    SO_IO1_reg = data[7-i];
    SI_IO0_reg = data[7-1-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    tx[1:0] = 2'b00;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end

`elsif SUHDCHK_CKFE
  for(i=0; i<8; i=i+2)
  begin
    CLK = 0;
    tx[1:0] = 2'b11;
    SO_IO1_reg = data[7-i];
    SI_IO0_reg = data[7-1-i];
    #(tCL);
    CLK = 1;
    #(tCH)
    tx[1:0] = 2'b00;
    CLK = 0;
  end

`else
  tx[1:0] = 2'b11;
  for(i=0; i<8; i=i+2)
  begin
    SO_IO1_reg = data[7-i];
    SI_IO0_reg = data[7-1-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end
  tx[1:0] = 2'b0;
`endif

end
endtask 
/////////////////////////////////////////////////////////////////
// Send Quad byte
/////////////////////////////////////////////////////////////////
task SendByteQuad;
input [7:0] data;
integer i;
begin
  CLK = 0;
  
`ifdef SUHDCHK_ZHD
  for(i=0; i<8; i=i+4)
  begin
    tx = 4'b1111;
    IO3_reg = data[7-i];
    WPn_IO2_reg = data[7-1-i];
    SO_IO1_reg = data[7-2-i];
    SI_IO0_reg = data[7-3-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    tx = 4'b0000;
    IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end

`elsif SUHDCHK_CKFE
  for(i=0; i<8; i=i+4)
  begin
    CLK = 0;
    tx = 4'b1111;
    IO3_reg = data[7-i];
    WPn_IO2_reg = data[7-1-i];
    SO_IO1_reg = data[7-2-i];
    SI_IO0_reg = data[7-3-i];
    #(tCL);
    CLK = 1;
    #(tCH)
    tx = 4'b0000;
    CLK = 0;
  end

`else
  tx = 4'b1111;
  for(i=0; i<8; i=i+4)
  begin
    IO3_reg = data[7-i];
    WPn_IO2_reg = data[7-1-i];
    SO_IO1_reg = data[7-2-i];
    SI_IO0_reg = data[7-3-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end
  tx = 4'b0000;
`endif
end
endtask 
/////////////////////////////////////////////////////////////////
// Send byte DDR
////////////////////////////////////////////////////////////////
task SendByte_DDR;
input [7:0] data;
integer i;
begin
  CLK = 0;

`ifdef SUHDCHK_ZHD
  for(i=0; i<8; i=i+2)
  begin
    tx[0] = 1;
    SI_IO0_reg = data[7-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    tx[0] = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD-tSU);
    tx[0] = 1;
    SI_IO0_reg = data[6-i];
    #(tSU);
    CLK = 0;
    #(tHD)
    tx[0] = 0;
    SI_IO0_reg = 0;
    #(tCL-tSU-tHD);
  end

`elsif SUHDCHK_CKFE
  tx[0] = 1;
  for(i=0; i<8; i=i+2)
  begin
    CLK = 0;
    #(tCL/2.0)
    SI_IO0_reg = data[7-i];
    #(tCL/2.0);
    CLK = 1;
    #(tCH/2.0)
    SI_IO0_reg = data[6-i];
    #(tCH/2.0)
    CLK = 0;
  end
  tx[0] = 0;

`else
  tx[0] = 1;
  for(i=0; i<8; i=i+2)
  begin
    SI_IO0_reg = data[7-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    SI_IO0_reg = data[6-i];
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end
  tx[0] = 0;
`endif

end
endtask 
/////////////////////////////////////////////////////////////////
// Send Dual byte DDR
/////////////////////////////////////////////////////////////////
task SendByteDual_DDR;
input [7:0] data;
integer i;
begin
  CLK = 0;
`ifdef SUHDCHK_ZHD
  for(i=0; i<8; i=i+4)
  begin
    tx[1:0] = 2'b11;
    SO_IO1_reg = data[7-i];
    SI_IO0_reg = data[6-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    tx[1:0] = 2'b00;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD-tSU);
    tx[1:0] = 2'b11;
    SO_IO1_reg = data[5-i];
    SI_IO0_reg = data[4-i];
    #(tSU);
    CLK = 0;
    #(tHD)
    tx[1:0] = 2'b00;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCL-tSU-tHD);
  end

`elsif SUHDCHK_CKFE
  tx[1:0] = 2'b11;
  for(i=0; i<8; i=i+4)
  begin
    CLK = 0;
    #(tCL/2.0)
    SO_IO1_reg = data[7-i];
    SI_IO0_reg = data[6-i];
    #(tCL/2.0);
    CLK = 1;
    #(tCH/2.0)
    SO_IO1_reg = data[5-i];
    SI_IO0_reg = data[4-i];
    #(tCH/2.0)
    CLK = 0;
  end
  tx[1:0] = 2'b00;

`else
  tx[1:0] = 2'b11;
  for(i=0; i<8; i=i+4)
  begin
    SO_IO1_reg = data[7-i];
    SI_IO0_reg = data[6-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    SO_IO1_reg = data[5-i];
    SI_IO0_reg = data[4-i];
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end
  tx[1:0] = 2'b0;
`endif

end
endtask 
/////////////////////////////////////////////////////////////////
// Send Quad byte DDR
/////////////////////////////////////////////////////////////////
task SendByteQuad_DDR;
input [7:0] data;
integer i;
begin
  CLK = 0;
  
`ifdef SUHDCHK_ZHD
    tx = 4'b1111;
    IO3_reg = data[7];
    WPn_IO2_reg = data[6];
    SO_IO1_reg = data[5];
    SI_IO0_reg = data[4];
    #(tSU);
    CLK = 1;
    #(tHD)
    tx = 4'b0000;
    IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD-tSU);
    tx = 4'b1111;
    IO3_reg = data[3];
    WPn_IO2_reg = data[2];
    SO_IO1_reg = data[1];
    SI_IO0_reg = data[0];
    #(tSU);
    CLK = 0;
    #(tHD)
    tx = 4'b0000;
    IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCL-tSU-tHD);

`elsif SUHDCHK_CKFE
    tx = 4'b1111;
    CLK = 0;
    #(tCL/2.0)
    IO3_reg = data[7];
    WPn_IO2_reg = data[6];
    SO_IO1_reg = data[5];
    SI_IO0_reg = data[4];
    #(tCL/2.0);
    CLK = 1;
    #(tCH/2.0)
    IO3_reg = data[3];
    WPn_IO2_reg = data[2];
    SO_IO1_reg = data[1];
    SI_IO0_reg = data[0];
    #(tCH/2.0);
    CLK = 0;
    tx = 4'b0000;

`else
  tx = 4'b1111;
    IO3_reg = data[7];
    WPn_IO2_reg = data[6];
    SO_IO1_reg = data[5];
    SI_IO0_reg = data[4];
    #(tSU);
    CLK = 1;
    #(tHD)
    IO3_reg = data[3];
    WPn_IO2_reg = data[2];
    SO_IO1_reg = data[1];
    SI_IO0_reg = data[0];
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  tx = 4'b0000;
`endif
end
endtask 
/////////////////////////////////////////////////////////////////
// Task to receive byte
/////////////////////////////////////////////////////////////////
task RcvByte;
output [7:0] data;
integer i;
begin
  CLK = 0;
  tx=0;
`ifdef SUHDCHK_CKFE
  for(i=0; i<8; i=i+1)
  begin
    CLK = 0;
    #(tCL)
    CLK = 1;
    #(tCH)
    CLK = 0;
    data[7-i] = SO_IO1;
  end
`else
  for(i=0; i<8; i=i+1)
  begin
    SI_IO0_reg = 0;
    #(tSU);
    CLK = 1;
    #(tHD)
    SI_IO0_reg = 0;
    #(tCH-tHD);
    SO_IO1_reg = 1'bz;
    SI_IO0_reg = 0;
    data[7-i] = SO_IO1;
    CLK = 0;
    #(tCL-tSU);
  end
`endif
end
endtask 
/////////////////////////////////////////////////////////////////
// Task to receive Dualbyte
/////////////////////////////////////////////////////////////////
task RcvByteDual;
output [7:0] data;
integer i;
begin
  CLK = 0;
  tx=0;
`ifdef SUHDCHK_CKFE
  for(i=0; i<8; i=i+2)
  begin
    CLK = 0;
    #(tCL)
    CLK = 1;
    #(tCH)
    CLK = 0;
    data[7-i] = SO_IO1;
    data[7-1-i] = SI_IO0;
  end
`else
  for(i=0; i<8; i=i+2)
  begin
    SO_IO1_reg = 0; 
    SI_IO0_reg = 0;    
    #(tSU);
    CLK = 1;
    #(tHD)
    SO_IO1_reg = 0;       
    SI_IO0_reg = 0;   
    #(tCH-tHD);
    SO_IO1_reg=1'bz;
    SI_IO0_reg=1'bz;  
    data[7-i] = SO_IO1;
    data[7-1-i] = SI_IO0;
    CLK = 0;
    #(tCL-tSU);
  end
`endif
end
endtask 
/////////////////////////////////////////////////////////////////
// Task to receive Quadbyte
/////////////////////////////////////////////////////////////////
task RcvByteQuad;
output [7:0] data;
integer i;
begin
  CLK = 0;
  tx=0;
`ifdef SUHDCHK_CKFE
  for(i=0; i<8; i=i+4)
  begin
    CLK = 0;
    #(tCL)
    CLK = 1;
    #(tCH)
    CLK = 0;
    data[7-i]= IO3;
    data[7-1-i]= WPn_IO2;
    data[7-2-i]= SO_IO1;
    data[7-3-i]= SI_IO0;
  end
`else
  for(i=0; i<8; i=i+4)
  begin
    IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0; 
    SI_IO0_reg = 0; 
    #(tSU);
    CLK = 1;
    #(tHD)
    IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD);
    IO3_reg=1'bz;
    WPn_IO2_reg=1'bz;
    SO_IO1_reg=1'bz;
    SI_IO0_reg=1'bz;
    data[7-i]= IO3;
    data[7-1-i]= WPn_IO2;
    data[7-2-i]= SO_IO1;
    data[7-3-i]= SI_IO0;
    CLK = 0;
    #(tCL-tSU);
  end
`endif
end
endtask 
/////////////////////////////////////////////////////////////////
// Task to receive byte DDR
/////////////////////////////////////////////////////////////////
task RcvByte_DDR;
output [7:0] data;
integer i;
begin
  CLK = 0;
  tx=0;
`ifdef SUHDCHK_CKFE
  for(i=0; i<8; i=i+2)
  begin
    CLK = 0;
    #(tCL)
    CLK = 1;
    data[7-i] = SO_IO1;
    #(tCH)
    CLK = 0;
    data[6-i] = SO_IO1;
  end
`else
  for(i=0; i<8; i=i+2)
  begin
    SI_IO0_reg = 0;
    #(tSU);
    CLK = 1;
    data[7-i] = SO_IO1;
    #(tHD)
    SI_IO0_reg = 0;
    #(tCH-tHD);
    SO_IO1_reg = 1'bz;
    SI_IO0_reg = 0;
    data[6-i] = SO_IO1;
    CLK = 0;
    #(tCL-tSU);
  end
`endif
end
endtask 
/////////////////////////////////////////////////////////////////
// Task to receive Dualbyte DDR
/////////////////////////////////////////////////////////////////
task RcvByteDual_DDR;
output [7:0] data;
integer i;
begin
  CLK = 0;
  tx=0;
`ifdef SUHDCHK_CKFE
  for(i=0; i<8; i=i+4)
  begin
    CLK = 0;
    #(tCL)
    CLK = 1;
    data[7-i] = SO_IO1;
    data[6-i] = SI_IO0;
    #(tCH)
    CLK = 0;
    data[5-i] = SO_IO1;
    data[4-i] = SI_IO0;
  end
`else
  for(i=0; i<8; i=i+4)
  begin
    SO_IO1_reg = 0; 
    SI_IO0_reg = 0;    
    #(tSU);
    CLK = 1;
    data[7-i] = SO_IO1;
    data[6-i] = SI_IO0;
    #(tHD)
    SO_IO1_reg = 0;       
    SI_IO0_reg = 0;   
    #(tCH-tHD);
    SO_IO1_reg=1'bz;
    SI_IO0_reg=1'bz;  
    data[5-i] = SO_IO1;
    data[4-i] = SI_IO0;
    CLK = 0;
    #(tCL-tSU);
  end
`endif
end
endtask 
/////////////////////////////////////////////////////////////////
// Task to receive Quadbyte DDR
/////////////////////////////////////////////////////////////////
task RcvByteQuad_DDR;
output [7:0] data;
integer i;
begin
  CLK = 0;
  tx=0;
`ifdef SUHDCHK_CKFE
    CLK = 0;
    #(tCL)
    CLK = 1;
    data[7]= IO3;
    data[6]= WPn_IO2;
    data[5]= SO_IO1;
    data[4]= SI_IO0;
    #(tCH)
    CLK = 0;
    data[3]= IO3;
    data[2]= WPn_IO2;
    data[1]= SO_IO1;
    data[0]= SI_IO0;
`else
    IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0; 
    SI_IO0_reg = 0; 
    #(tSU);
    CLK = 1;
    data[7]= IO3;
    data[6]= WPn_IO2;
    data[5]= SO_IO1;
    data[4]= SI_IO0;
    #(tHD)
    IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;
    #(tCH-tHD);
    IO3_reg=1'bz;
    WPn_IO2_reg=1'bz;
    SO_IO1_reg=1'bz;
    SI_IO0_reg=1'bz;
    data[3]= IO3;
    data[2]= WPn_IO2;
    data[1]= SO_IO1;
    data[0]= SI_IO0;
    CLK = 0;
  end
`endif
end
endtask 
/////////////////////////////////////////////////////////////////
// Send command
/////////////////////////////////////////////////////////////////
task Cmd;
input [2:0] Mode;
input [7:0] data;
begin
	if (Mode == 3'h1)
	begin
	SendByte(data);
	end
     
   	else if (Mode == 3'h2)
	begin
	SendByteDual(data);
	end
          
        else if (Mode == 3'h4)
	begin
	SendByteQuad(data);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Send 32-bit address
/////////////////////////////////////////////////////////////////
task Addr_32bit;
input [2:0] Mode;
input [31:0] addr;
begin
	if (Mode == 3'h1)
	begin
	SendByte(addr[31:24]);
	SendByte(addr[23:16]);
	SendByte(addr[15:8]);
	SendByte(addr[7:0]);
	end

	else if (Mode == 3'h2)
	begin
	SendByteDual(addr[31:24]);
	SendByteDual(addr[23:16]);
	SendByteDual(addr[15:8]);
	SendByteDual(addr[7:0]);
	end

     	else if (Mode == 3'h4)
	begin
	SendByteQuad(addr[31:24]);
	SendByteQuad(addr[23:16]);
	SendByteQuad(addr[15:8]);
	SendByteQuad(addr[7:0]);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Send 32-bit address DDR
/////////////////////////////////////////////////////////////////
task Addr_32bit_DDR;
input [2:0] Mode;
input [31:0] addr;
begin
  if (Mode == 3'h1)
  begin
    SendByte_DDR(addr[31:24]);
    SendByte_DDR(addr[23:16]);
    SendByte_DDR(addr[15:8]);
    SendByte_DDR(addr[7:0]);
  end

  else if (Mode == 3'h2)
  begin
    SendByteDual_DDR(addr[31:24]);
    SendByteDual_DDR(addr[23:16]);
    SendByteDual_DDR(addr[15:8]);
    SendByteDual_DDR(addr[7:0]);
  end

  else if (Mode == 3'h4)
  begin
    SendByteQuad_DDR(addr[31:24]);
    SendByteQuad_DDR(addr[23:16]);
    SendByteQuad_DDR(addr[15:8]);
    SendByteQuad_DDR(addr[7:0]);
  end
end
endtask
/////////////////////////////////////////////////////////////////
// Send 24-bit address
/////////////////////////////////////////////////////////////////
task Addr_24bit;
input [2:0] Mode;
input [23:0] addr;
begin
  if (Mode == 3'h1)
  begin
    SendByte(addr[23:16]);
    SendByte(addr[15:8]);
    SendByte(addr[7:0]);
  end

  else if (Mode == 3'h2)
  begin
    SendByteDual(addr[23:16]);
    SendByteDual(addr[15:8]);
    SendByteDual(addr[7:0]);
  end

  else if (Mode == 3'h4)
  begin
    SendByteQuad(addr[23:16]);
    SendByteQuad(addr[15:8]);
    SendByteQuad(addr[7:0]);
  end
end
endtask
/////////////////////////////////////////////////////////////////
// Send 24-bit address DDR
/////////////////////////////////////////////////////////////////
task Addr_24bit_DDR;
input [2:0] Mode;
input [23:0] addr;
begin
  if (Mode == 3'h1)
  begin
    SendByte_DDR(addr[23:16]);
    SendByte_DDR(addr[15:8]);
    SendByte_DDR(addr[7:0]);
  end

  else if (Mode == 3'h2)
  begin
    SendByteDual_DDR(addr[23:16]);
    SendByteDual_DDR(addr[15:8]);
    SendByteDual_DDR(addr[7:0]);
  end

  else if (Mode == 3'h4)
  begin
    SendByteQuad_DDR(addr[23:16]);
    SendByteQuad_DDR(addr[15:8]);
    SendByteQuad_DDR(addr[7:0]);
  end
end
endtask
/////////////////////////////////////////////////////////////////
// Task to send mode byte
/////////////////////////////////////////////////////////////////
task ModeByte;
input [2:0] Mode;
input [7:0] data;
begin
	if (Mode == 3'h1)
	begin
	SendByte(data);
	end

	else if (Mode == 3'h2)
	begin
	SendByteDual(data);
	end
      
        else if (Mode == 3'h4)
	begin
	SendByteQuad(data);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Task to send mode byte DDR
/////////////////////////////////////////////////////////////////
task ModeByte_DDR;
input [2:0] Mode;
input [7:0] data;
begin
  if (Mode == 3'h1)
  begin
    SendByte_DDR(data);
  end

  else if (Mode == 3'h2)
  begin
    SendByteDual_DDR(data);
  end

  else if (Mode == 3'h4)
  begin
    SendByteQuad_DDR(data);
  end
end
endtask
/////////////////////////////////////////////////////////////////
// Task to send latency
/////////////////////////////////////////////////////////////////
task Latency;
input [7:0] num;
integer i;
begin
  CLK = 0;
  for(i=0; i<num; i=i+1)
  begin
`ifdef SUHDCHK_CKFE
    CLK = 0;
    #(tCL);
    CLK = 1;
    #(tCH);
    CLK = 0;
`else
    IO3_reg =1'bz;
    WPn_IO2_reg =1'bz;
    SO_IO1_reg = 1'bz;
    SI_IO0_reg = 1'bz;
    #(tSU);
    CLK = 1;
    #(tHD)
    IO3_reg =1'bz;
    WPn_IO2_reg =1'bz;
    SO_IO1_reg = 1'bz;
    SI_IO0_reg = 1'bz;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
`endif
  end
end
endtask
/////////////////////////////////////////////////////////////////
// SPI enable for Modes
/////////////////////////////////////////////////////////////////
task SPIEnable;
input [2:0] Mode;
begin
    SelectChip();
    Cmd(Mode,8'hff);
    DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// DPI enable for Modes
/////////////////////////////////////////////////////////////////
task DPIEnable;
input [2:0] Mode;
begin
    SelectChip();
    Cmd(Mode,8'h37);
    DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// QPI enable for Modes
/////////////////////////////////////////////////////////////////
task QPIEnable;
input [2:0] Mode;
begin
    SelectChip();
    Cmd(Mode,8'h38);
    DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Write enable
/////////////////////////////////////////////////////////////////
task WriteEnable;
input [2:0] Mode;
begin
    	SelectChip();
    	Cmd(Mode,8'h06);
    	DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// SPI enable
/////////////////////////////////////////////////////////////////
task SPI_Enable;
input [2:0] Mode;
begin
    	SelectChip();
    	Cmd(Mode,8'hff);
    	DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// DPI enable
/////////////////////////////////////////////////////////////////
task DPI_Enable;
input [2:0] Mode;
begin
    	SelectChip();
    	Cmd(Mode,8'h37);
    	DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// QPI enable
/////////////////////////////////////////////////////////////////
task QPI_Enable;
input [2:0] Mode;
begin
    	SelectChip();
    	Cmd(Mode,8'h38);
    	DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Write disable
/////////////////////////////////////////////////////////////////
task WriteDisable;
input [2:0] Mode;
begin
	SelectChip();
    	Cmd(Mode,8'h04);
    	DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
//  Write status
/////////////////////////////////////////////////////////////////
task WriteStatus;
input [2:0] Mode;
input [7:0]data;
begin
    	SelectChip();
    	Cmd(Mode,8'h01);
	
    	SendByte(data);

    	DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Write Any Register
/////////////////////////////////////////////////////////////////
task WriteReg;
input [2:0] Mode;
input [31:0] addr;
input [31:0] data;

begin
  $display("[INFO] Write Any Register : Address %x data %x", addr, data);
  case(addr)
    32'h0: $display("       --> Status Register");
    32'h1: $display("       --> Interrupt Status Register");
    32'h2: $display("       --> Config Register 1");
    32'h3: $display("       --> Config Register 2");
    32'h4: $display("       --> Interrupt Config Register");
    32'h5: $display("       --> ECC Test - Data Input Register");
    32'h6: $display("       --> ECC Test - Error Injection Register");
    32'h7: $display("       --> ECC Test - Data Output Register");
    32'h8: $display("       --> ECC Test - Error Count Register");
    32'h30: $display("       --> Device Identification Register");
    32'h40: $display("       --> Device Unique Identification Register");
    32'h50: $display("       --> Serial Number Register");
    default: $display("       --> Invalid Register address");
  endcase

  SelectChip();
  Cmd(Mode, 8'h71);
  Addr_32bit(Mode, addr);

  if (Mode == 3'h1)
  begin
    case(addr)
      32'h5, 32'h6: begin
        SendByte(data[31:24]);
        SendByte(data[23:16]);
        SendByte(data[15:8]);
        SendByte(data[7:0]);
      end
      default: begin
        SendByte(data[7:0]);
      end
    endcase
  end

  DeSelChip();
end
endtask

task WriteAnyRegister;
input [2:0] Mode;
input [31:0] addr;
input [7:0] data1;

begin
   	SelectChip();
   	Cmd(Mode,8'h71);

  	SendByte(addr[31:24]);
   	SendByte(addr[23:16]);
   	SendByte(addr[15:8]);
   	SendByte(addr[7:0]);

   	SendByte(data1);
 

   	DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Read status
/////////////////////////////////////////////////////////////////
task ReadStatus;
input [2:0] Mode;
input [7:0] data;
reg [7:0] data_rcv;
begin
    	SelectChip();
    	Cmd(Mode,8'h05);
	
    	RcvByte(data_rcv);

    	DeSelChip();
	
	if((data_rcv & 8'hfc) !== (data & 8'hfc))
		$display("[ERROR] Status Compare Error : Expected %x Received %x",data,data_rcv);
	else
		$display("[INFO] Status Compare Pass : Expected %x Received %x",data,data_rcv);
end
endtask
/////////////////////////////////////////////////////////////////
// Read Any Register
/////////////////////////////////////////////////////////////////
task ReadReg;
input [2:0] Mode;
input [31:0] addr;
input [7:0] data;
input [7:0] latency_val;
reg [7:0] data_rcv;
reg error_flag;
integer i;
reg [31:0] id;
reg [63:0] uid;
begin

  $display("[INFO] Read Any Register : Address %x data %x", addr, data);
  case(addr)
    32'h0: $display("       --> Status Register");
    32'h1: $display("       --> Interrupt Status Register");
    32'h2: $display("       --> Config Register 1");
    32'h3: $display("       --> Config Register 2");
    32'h4: $display("       --> Interrupt Config Register");
    32'h5: $display("       --> ECC Test - Data Input Register");
    32'h6: $display("       --> ECC Test - Error Injection Register");
    32'h7: $display("       --> ECC Test - Data Output Register");
    32'h8: $display("       --> ECC Test - Error Count Register");
    32'h30: $display("       --> Device Identification Register");
    32'h40: $display("       --> Device Unique Identification Register");
    32'h50: $display("       --> Serial Number Register");
    default: $display("       --> Invalid Register address");
  endcase

  SelectChip();
  Cmd(Mode, 8'h65);
  Addr_32bit(Mode, addr);

  case(addr)
    // 32-bit output
    32'h5, 32'h6, 32'h7, 32'h8, 32'h30:
    begin
      Latency(latency_val);
      for (i=0; i<4; i=i+1) begin
        RcvByte(data_rcv);
        id = {id[23:0], data_rcv};
        //id[31-(i*8) : 32-(i*8)] = data_rcv;
      end
    end

    // 64-bit output 
    32'h40, 32'h50:
    begin
      Latency(latency_val);
      for (i=0; i<8; i=i+1) begin
        RcvByte(data_rcv);
        uid = {uid[55:0], data_rcv};
        //uid[63-(i*8) : 63-(i*8)] = data_rcv;
      end
    end

    // 8-bit output 
    default: begin
      Latency(latency_val);
      RcvByte(data_rcv);
    end
  endcase

  DeSelChip();

  error_flag = 0;
  case(addr)
    32'h0:
      if (data_rcv !== (data & 8'hfc))
        error_flag = 1'b1;
    32'h2: //CR1
      if ((data_rcv & 8'hE7) !== (data & 8'hE7))
        error_flag = 1'b1;
    32'h3: //CR2
      if ((data_rcv & 8'h0f) !== (data & 8'h0f))
        error_flag = 1'b1;
    32'h4: //INTCR
      if ((data_rcv & 8'h6f) !== (data & 8'h6f))
        error_flag = 1'b1;
    32'h1: //ISR
      if ((data_rcv & 8'hff) !== (data & 8'hff))
        error_flag = 1'b1;
    //32'h5, 32'h6: // EEC reg
    //  if (data_rcv !== data)
    //    error_flag = 1'b1;

  endcase

  if (error_flag)
    $display("[ERROR] Register Compare Error (Reg 0x%x) : Expected 0x%x Received 0x%x (time:%t)", addr, data, data_rcv, $time);
  else
    case(addr)
      32'h0, 32'h1, 32'h2, 32'h3, 32'h4:
        $display("[INFO] Register Compare Pass (Reg 0x%x) : Expected 0x%x Received 0x%x", addr, data, data_rcv);
      32'h5, 32'h6, 32'h7, 32'h8:
        $display("[INFO] Register Read ECC register : 0x%x", id);
      32'h30:
        $display("[INFO] Register Read ID : 0x%x", id);
      32'h40:
        $display("[INFO] Register Read UID : 0x%x", uid);
      32'h50:
        $display("[INFO] Register Read SN : 0x%x", uid);
      default:
        $display("[ERROR] Register Compare not performed due to invalid/unsupported address (Reg 0x%x)", addr);
    endcase
end
endtask
task ReadAnyRegister;
input [2:0] Mode;
input [31:0] addr;
input [7:0] data1;
reg [7:0] data_rcv1;

begin
   	SelectChip();
   	Cmd(Mode,8'h65);

  	SendByte(addr[31:24]);
   	SendByte(addr[23:16]);
   	SendByte(addr[15:8]);
   	SendByte(addr[7:0]);
   	SendByte(8'h00);
 
   	RcvByte(data_rcv1);

    	DeSelChip();

	
		if(data_rcv1 !== data1)
              
			$display("[ERROR] CFG Reg Compare Error : Expected %x Received %x ",data1, data_rcv1);
		else
			$display("[INFO] CFG Reg Compare Pass : Expected %x Received %x ",data1, data_rcv1);

   
end
endtask

task ReadID;



reg [31:0] data_rcv1;

begin
 	SelectChip();
 	Cmd(3'h1,8'h9f);


 	RcvByte(data_rcv1[31:24]);
 	RcvByte(data_rcv1[23:16]);
 	RcvByte(data_rcv1[15:8]);
 	RcvByte(data_rcv1[7:0]);
 	DeSelChip();

	$display("[INFO] Read ID Received 0x%x ",data_rcv1);
  $display("[INFO] Manufacturer ID : 0x%x ", data_rcv1[31:24]);
  $display("[INFO] Interface       : 0x%x ", data_rcv1[23:20]);
  $display("[INFO] Voltage         : 0x%x ", data_rcv1[19:16]);
  $display("[INFO] Temperature     : 0x%x ", data_rcv1[15:12]);
  $display("[INFO] Frequency       : 0x%x ", data_rcv1[7:0]);
  case(data_rcv1[11:8])
    4'b1000: $display("[INFO] Density         : 1Gb (512Mb per lane)");
    4'b1001: $display("[INFO] Density         : 2Gb (1Gb per lane)");
    4'b1010: $display("[INFO] Density         : 4Gb (2Gb per lane)");
    4'b1100: $display("[INFO] Density         : 8Gb (4Gb per lane)");
    default: $display("[INFO] Unknown Density");
  endcase

   
end
endtask


task ReadConfig;
input [2:0] Mode;
input [7:0] data1;
input [7:0] data2;
begin

ReadAnyRegister(3'h1, 32'h2, data1);
ReadAnyRegister(3'h1, 32'h3, data2);
end
endtask



/////////////////////////////////////////////////////////////////
// Memory Write
/////////////////////////////////////////////////////////////////
task Write;
input [2:0] Mode;
input [23:0] addr;
input [7:0] data;
input [23:0] num_data_i;
integer i; 
begin
	$display("[INFO] Mem write  : Address %x data %x num_data %x",addr, data, num_data_i);
    	SelectChip();
    	Cmd(Mode,8'h02);
    	Addr_32bit(Mode,addr);
	for (i=0;i<num_data_i;i=i+1)
	begin
	
	if (Mode == 3'h1)
	begin
	SendByte(data);
	end
        
        else if (Mode == 3'h4)
	begin
	SendByteQuad(data);
	end

 	end
    	DeSelChip();
end
endtask

task WriteSeq;
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [23:0] num_data_i;
reg [7:0] data;
integer i; 
begin
  $display("[INFO] Mem write (sequential)  : Address %x num_data %x data_base %x data_inc %x",addr, num_data_i, data_base, data_inc);
  data = data_base;
  SelectChip();
  Cmd(Mode,8'h02);
  Addr_32bit(Mode,addr);
  for (i=0; i<num_data_i; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      SendByte(data);
    end
    else if (Mode == 3'h2)
    begin
      SendByteDual(data);
    end
    else if (Mode == 3'h4)
    begin
      SendByteQuad(data);
    end

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task FastWriteSeqXIP();
input [2:0] Mode;
input [23:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [31:0] num_data_i;
input xip_on;
input [7:0] mode_byte;
reg [7:0] data;
integer i; 
begin
  $display("[INFO] Fast Mem write (sequential)  : Address %x num_data %x data_base %x data_inc %x",addr, num_data_i, data_base, data_inc);
  data = data_base;
  SelectChip();
  if (!xip_on) begin
    Cmd(Mode,8'hDA);
  end
  Addr_32bit(Mode,addr);
  ModeByte(Mode,mode_byte);
  for (i=0; i<num_data_i; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      SendByte(data);
    end
    else if (Mode == 3'h2)
    begin
      SendByteDual(data);
    end
    else if (Mode == 3'h4)
    begin
      SendByteQuad(data);
    end

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task WriteQuadIOSeqXIP();
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [23:0] num_data_i;
input xip_on;
input [7:0] mode_byte;
reg [7:0] data;
integer i; 
begin
  $display("[INFO] Write Quad I/O (sequential)  : Address %x num_data %x data_base %x data_inc %x",addr, num_data_i, data_base, data_inc);
  data = data_base;
  SelectChip();
  if (!xip_on) begin
    Cmd(Mode,8'hd2);
  end
  Addr_32bit(3'h4,addr);
  ModeByte(3'h4,mode_byte);
  for (i=0; i<num_data_i; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      SendByteQuad(data);
    end

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task DDRFastWriteSeq;
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [23:0] num_data_i;
input xip_on;
input [7:0] mode_byte;
reg [7:0] data;
integer i; 
begin
  $display("[INFO] Fast Mem write DDR (sequential)  : Address %x num_data %x data_base %x data_inc %x",addr, num_data_i, data_base, data_inc);
  data = data_base;
  SelectChip();
  if (!xip_on) begin
    Cmd(Mode,8'hDE);
  end
  Addr_32bit_DDR(Mode,addr);
  ModeByte_DDR(Mode,mode_byte);
  for (i=0; i<num_data_i; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      SendByte_DDR(data);
    end
    //else if (Mode == 3'h2)
    //begin
    //  SendByteDual_DDR(data);
    //end
    //else if (Mode == 3'h4)
    //begin
    //  SendByteQuad_DDR(data);
    //end

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task DDRWriteQuadIOSeqXIP();
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [23:0] num_data_i;
input xip_on;
input [7:0] mode_byte;
reg [7:0] data;
integer i; 
begin
  $display("[INFO] Write Quad I/O (sequential)  : Address %x num_data %x data_base %x data_inc %x",addr, num_data_i, data_base, data_inc);
  data = data_base;
  SelectChip();
  if (!xip_on) begin
    Cmd(Mode,8'hd1);
  end
  Addr_32bit_DDR(3'h4,addr);
  ModeByte_DDR(3'h4,mode_byte);
  for (i=0; i<num_data_i; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      SendByteQuad_DDR(data);
    end

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

/////////////////////////////////////////////////////////////////
// Memory Read
/////////////////////////////////////////////////////////////////
task Read;
input [2:0] Mode;
input [23:0] addr;
input [7:0] data;
input [7:0] latency_val;
input [23:0] num_data_o;
integer i;
reg [7:0] data_rcv;
begin
	$display("[INFO] Mem read  : Address %x data %x latency %x num_data %x",addr, data, latency_val, num_data_o);
    	SelectChip();
    	Cmd(Mode,8'h03);
    	Addr_32bit(Mode,addr);
    	Latency(latency_val);
    	for (i=0;i<num_data_o;i=i+1)
	begin
           
	if (Mode == 3'h1)
	begin
	RcvByte(data_rcv);
	end
	else if (Mode == 3'h4)
	begin
	RcvByteQuad(data_rcv);
	end

		if(data_rcv !== data)
			$display("[ERROR] Mem Read Compare Error : Expected %x Received %x",data,data_rcv);
		else
			$display("[INFO] Mem Read Compare Pass : Expected %x Received %x",data,data_rcv);
	end
    	DeSelChip();
end
endtask

task ReadSeq;
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
//input [7:0] latency_val;
input [23:0] num_data_o;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  //$display("[INFO] Mem read (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  $display("[INFO] Mem read (sequential) : Address %x num_data %x data_base %x data_inc %x",addr, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  Cmd(Mode, 8'h13);
  Addr_32bit(Mode, addr);
  //Latency(latency_val);

  for (i=0; i<num_data_o; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      RcvByte(data_rcv);
    end
    else if (Mode == 3'h2)
    begin
      RcvByteDual(data_rcv);
    end
    else if (Mode == 3'h4)
    begin
      RcvByteQuad(data_rcv);
    end

    if (data_rcv !== data)
    $display("[ERROR] Mem Read Compare Error : Expected %x Received %x",data,data_rcv);
    else
    $display("[INFO] Mem Read Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();

end
endtask

task ReadSeq_24bit;
input [2:0] Mode;
input [23:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
//input [7:0] latency_val;
input [23:0] num_data_o;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  //$display("[INFO] Mem read (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  $display("[INFO] Mem read (sequential) : Address %x num_data %x data_base %x data_inc %x",addr, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  Cmd(Mode, 8'h03);
  Addr_24bit(Mode, addr);
  //Latency(latency_val);

  for (i=0; i<num_data_o; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      RcvByte(data_rcv);
    end
    else if (Mode == 3'h2)
    begin
      RcvByteDual(data_rcv);
    end
    else if (Mode == 3'h4)
    begin
      RcvByteQuad(data_rcv);
    end

    if (data_rcv !== data)
    $display("[ERROR] Mem Read Compare Error : Expected %x Received %x",data,data_rcv);
    else
    $display("[INFO] Mem Read Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();

end
endtask

/////////////////////////////////////////////////////////////////
// Memory Read using Fast Read
/////////////////////////////////////////////////////////////////
task FastRead;
input [2:0] Mode;
input [23:0] addr;
input [7:0] data;
input [7:0] latency_val;
input [23:0] num_data_o;
integer i;
reg [7:0] data_rcv;
begin
	$display("[INFO] Mem read  : Address %x data %x latency %x num_data %x",addr, data, latency_val, num_data_o);
    	SelectChip();
    	Cmd(Mode,8'h0B);
    	Addr_32bit(Mode,addr);
	ModeByte(Mode,8'hFF);
    	Latency(latency_val);
    	for (i=0;i<num_data_o;i=i+1)
	begin
        
	if (Mode == 3'h1)
	begin
	RcvByte(data_rcv);
	end

	else if (Mode == 3'h4)
	begin
	RcvByteQuad(data_rcv);
	end
	
		if(data_rcv !== data)
			$display("[ERROR] Mem Fast Read Compare Error : Expected %x Received %x",data,data_rcv);
		else
			$display("[INFO] Mem Fast Read Compare Pass : Expected %x Received %x",data,data_rcv);
	end
    	DeSelChip();
end
endtask

task FastReadSeqXIP();
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [7:0] latency_val;
input [23:0] num_data_o;
//input xip_on;
//input [7:0] mode_byte;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  $display("[INFO] Mem Fast read (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  //if (!xip_on) begin
    Cmd(Mode,8'h0C);
  //end
  Addr_32bit(Mode,addr);
  //ModeByte(Mode,mode_byte);
  Latency(latency_val);
  for (i=0;i<num_data_o;i=i+1)
  begin
    if (Mode == 3'h1)
    begin
    RcvByte(data_rcv);
    end
    else if (Mode == 3'h2)
    begin
      RcvByteDual(data_rcv);
    end
    else if (Mode == 3'h4)
    begin
      RcvByteQuad(data_rcv);
    end

    if(data_rcv !== data)
      $display("[ERROR] Mem Fast Read Compare Error : Expected %x Received %x",data,data_rcv);
    else
      $display("[INFO] Mem Fast Read Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task FastReadSeqXIP_24bit();
input [2:0] Mode;
input [23:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [7:0] latency_val;
input [23:0] num_data_o;
//input xip_on;
//input [7:0] mode_byte;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  $display("[INFO] Mem Fast read (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  //if (!xip_on) begin
    Cmd(Mode,8'h0B);
  //end
  Addr_24bit(Mode,addr);
  //ModeByte(Mode,mode_byte);
  Latency(latency_val);
  for (i=0;i<num_data_o;i=i+1)
  begin
    if (Mode == 3'h1)
    begin
    RcvByte(data_rcv);
    end
    else if (Mode == 3'h2)
    begin
      RcvByteDual(data_rcv);
    end
    else if (Mode == 3'h4)
    begin
      RcvByteQuad(data_rcv);
    end

    if(data_rcv !== data)
      $display("[ERROR] Mem Fast Read Compare Error : Expected %x Received %x",data,data_rcv);
    else
      $display("[INFO] Mem Fast Read Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task ReadQuadOutputSeqXIP;
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [7:0] latency_val;
input [23:0] num_data_o;
//input xip_on;
//input [7:0] mode_byte;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  $display("[INFO] Mem Read Quad Output (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  //if (!xip_on) begin
    Cmd(Mode,8'h6C);
  //end
  Addr_32bit(Mode,addr);
  //ModeByte(Mode,mode_byte);
  Latency(latency_val);
  for (i=0;i<num_data_o;i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      RcvByteQuad(data_rcv);
    end

    if(data_rcv !== data)
      $display("[ERROR] Mem Read Quad Output Compare Error : Expected %x Received %x",data,data_rcv);
    else
      $display("[INFO] Mem Read Quad Output Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task ReadQuadOutputSeqXIP_24bit;
input [2:0] Mode;
input [23:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [7:0] latency_val;
input [23:0] num_data_o;
//input xip_on;
//input [7:0] mode_byte;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  $display("[INFO] Mem Read Quad Output (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  //if (!xip_on) begin
    Cmd(Mode,8'h6B);
  //end
  Addr_24bit(Mode,addr);
  //ModeByte(Mode,mode_byte);
  Latency(latency_val);
  for (i=0;i<num_data_o;i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      RcvByteQuad(data_rcv);
    end

    if(data_rcv !== data)
      $display("[ERROR] Mem Read Quad Output Compare Error : Expected %x Received %x",data,data_rcv);
    else
      $display("[INFO] Mem Read Quad Output Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task ReadQuadIOSeqXIP();
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [7:0] latency_val;
input [23:0] num_data_o;
input xip_on;
input [7:0] mode_byte;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  $display("[INFO] Mem Read Quad I/O (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  if (!xip_on) begin
    Cmd(Mode,8'hEB);
  end
  Addr_32bit(3'h4,addr);
  ModeByte(3'h4,mode_byte);
  Latency(latency_val);
  for (i=0;i<num_data_o;i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      RcvByteQuad(data_rcv);
    end

    if(data_rcv !== data)
      $display("[ERROR] Mem Read Quad I/O Compare Error : Expected %x Received %x",data,data_rcv);
    else
      $display("[INFO] Mem Read Quad I/O Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task DDRFastReadSeqXIP();
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [7:0] latency_val;
input [23:0] num_data_o;
input xip_on;
input [7:0] mode_byte;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  $display("[INFO] Mem Fast read DDR (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  if (!xip_on) begin
    Cmd(Mode,8'h0D);
  end
  Addr_32bit_DDR(Mode,addr);
  ModeByte_DDR(Mode,mode_byte);
  Latency(latency_val);
  for (i=0;i<num_data_o;i=i+1)
  begin
    if (Mode == 3'h1)
    begin
    RcvByte_DDR(data_rcv);
    end
    else if (Mode == 3'h2)
    begin
      RcvByteDual_DDR(data_rcv);
    end
    else if (Mode == 3'h4)
    begin
      RcvByteQuad_DDR(data_rcv);
    end

    if(data_rcv !== data)
      $display("[ERROR] Mem Fast Read DDR Compare Error : Expected %x Received %x",data,data_rcv);
    else
      $display("[INFO] Mem Fast Read DDR Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();
end
endtask

task DDRReadQuadIOSeqXIP();
input [2:0] Mode;
input [31:0] addr;
input [7:0] data_base;
input [7:0] data_inc;
input [7:0] latency_val;
input [23:0] num_data_o;
input xip_on;
input [7:0] mode_byte;
integer i;
reg [7:0] data_rcv;
reg [7:0] data;
begin
  $display("[INFO] Mem Read Quad I/O DDR (sequential) : Address %x latency %x num_data %x data_base %x data_inc %x",addr, latency_val, num_data_o, data_base, data_inc);
  data = data_base;
  SelectChip();
  if (!xip_on) begin
    Cmd(Mode,8'hED);
  end
  Addr_32bit_DDR(3'h4,addr);
  ModeByte_DDR(3'h4,mode_byte);
  Latency(latency_val);
  for (i=0;i<num_data_o;i=i+1)
  begin
    if (Mode == 3'h1)
    begin
    RcvByteQuad_DDR(data_rcv);
    end

    if(data_rcv !== data)
      $display("[ERROR] Mem Read Quad I/O DDR Compare Error : Expected %x Received %x",data,data_rcv);
    else
      $display("[INFO] Mem Read Quad I/O DDR Compare Pass : Expected %x Received %x",data,data_rcv);

    data = data + data_inc;
  end
  DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Status register write read check
/////////////////////////////////////////////////////////////////
task Status_Register_Check;
input [2:0] Mode;
begin 
       
	WriteEnable(3'h1);
   	WriteStatus(3'h1,8'hFF);
   	ReadStatus(3'h1,8'hFF);
   	
	WriteStatus(3'h1,8'haa);
   	ReadStatus(3'h1,8'hFF);
end
endtask
/////////////////////////////////////////////////////////////////
// Config register write read check
/////////////////////////////////////////////////////////////////
task Config_Register_Check;
input [2:0] Mode;
begin 

	$display("*****************************************************************");
	$display("** Config register write read check");
	$display("*****************************************************************");

   	WriteEnable(3'h1);
	WriteAnyRegister(3'h1, 32'h2, 8'hFF);
	WriteEnable(3'h1);
	WriteAnyRegister(3'h1, 32'h3, 8'hFF);
   	ReadConfig(3'h1,8'hFF,8'hAF);



	WriteAnyRegister(3'h1, 32'h2, 8'haa);
	WriteAnyRegister(3'h1, 32'h3, 8'haa);
   	ReadConfig(3'h1,8'hFF,8'hAF);

  	WriteEnable(3'h1);
  	WriteDisable(3'h1);
	WriteAnyRegister(3'h1, 32'h2, 8'haa);
	WriteAnyRegister(3'h1, 32'h3, 8'haa);
   	ReadConfig(3'h1,8'hFF,8'hAF);

  	WriteEnable(3'h1);
	WriteAnyRegister(3'h1, 32'h2, 8'h00);
	WriteEnable(3'h1);
	WriteAnyRegister(3'h1, 32'h3, 8'h08);
   	ReadConfig(3'h1,8'h00,8'h08);


end
endtask
/////////////////////////////////////////////////////////////////
// Write Read Memory check
/////////////////////////////////////////////////////////////////
task Memory_Write_Read_Check;
input [2:0] Mode;
begin 

        
	$display("*****************************************************************");
	$display("** Write Read Memory check ");
	$display("*****************************************************************");
        if(Mode == 3'h1)
	begin
   	WriteEnable(3'h1);
   	WriteStatus(3'h1,8'h00);
   	ReadStatus(3'h1,8'h00);

  	WriteEnable(3'h1);
	WriteAnyRegister(3'h1, 32'h2, 8'h00);
	WriteEnable(3'h1);
	WriteAnyRegister(3'h1, 32'h3, 8'h08);
   	ReadConfig(3'h1,8'h00,8'h08);

   	WriteEnable(3'h1);
   	Write(3'h1,24'h000000,8'ha3,8'h0f);
   	Read(3'h1,24'h000000,8'ha3,8'h08,8'h0f);
  	FastRead(3'h1,24'h000000,8'ha3,8'h08,8'h0f);
        end
   	  	
        else if(Mode == 3'h4)
	begin
        
   	WriteEnable(3'h4);
   	Write(3'h4,24'h000200,8'hc5,8'h0f);

   	//Read(3'h4,24'h000200,8'hc5,8'h08,8'h0f);
  	FastRead(3'h4,24'h000200,8'hc5,8'h08,8'h0f);

   	
        end
end
endtask


